#!/usr/bin/env python3
"""
TEI Pipeline Setup Script

Creates /Applications/TEI Pipeline.app with embedded Python env.
Usage: python3 setup_app.py
"""

import os
import sys
import shutil
import subprocess
import platform
import plistlib
from pathlib import Path

APP_NAME = "TEI Pipeline"
BUNDLE_ID = "com.tei-pipeline.app"
APP_VERSION = "2.0.0"
BACKEND_PORT = 5199

SCRIPT_DIR = Path(__file__).resolve().parent
REQUIRED_FILES = ["server.py", "__main__.py"]
REQUIRED_DIRS = ["core", "templates"]


def run(cmd, **kwargs):
    if isinstance(cmd, list):
        print(f"  > {' '.join(str(c) for c in cmd)}")
    else:
        print(f"  > {cmd}")
    subprocess.check_call(cmd, shell=isinstance(cmd, str), **kwargs)


def detect_platform():
    machine = platform.machine().lower()
    system = platform.system()
    if system == "Darwin" and machine == "arm64":
        return "mps"
    elif system == "Darwin":
        return "cpu"
    try:
        r = subprocess.run(["nvidia-smi"], capture_output=True, text=True)
        if r.returncode == 0:
            return "cuda"
    except FileNotFoundError:
        pass
    return "cpu"


def get_pytorch_install_args(gpu_platform):
    if gpu_platform == "mps":
        return ["torch", "torchvision", "torchaudio"]
    elif gpu_platform == "cuda":
        return ["torch", "torchvision", "torchaudio",
                "--index-url", "https://download.pytorch.org/whl/cu124"]
    return ["torch", "torchvision", "torchaudio",
            "--index-url", "https://download.pytorch.org/whl/cpu"]


def build_icon(resources_dir):
    """Build .icns from icon.png using Pillow + iconutil."""
    icon_png = SCRIPT_DIR / "icon.png"
    if not icon_png.exists():
        print("  No icon.png found, skipping icon.")
        return ""

    # Build the .iconset directory from the single PNG
    iconset_dir = resources_dir / "AppIcon.iconset"
    iconset_dir.mkdir(exist_ok=True)

    try:
        from PIL import Image
        img = Image.open(icon_png)
        sizes = {
            "icon_16x16.png": 16, "icon_16x16@2x.png": 32,
            "icon_32x32.png": 32, "icon_32x32@2x.png": 64,
            "icon_128x128.png": 128, "icon_128x128@2x.png": 256,
            "icon_256x256.png": 256, "icon_256x256@2x.png": 512,
            "icon_512x512.png": 512, "icon_512x512@2x.png": 1024,
        }
        for name, sz in sizes.items():
            # Clamp to source size
            actual = min(sz, img.size[0])
            img.resize((actual, actual), Image.LANCZOS).save(iconset_dir / name)
    except ImportError:
        # Pillow not available at setup time — copy source sizes from iconset if present
        src_iconset = SCRIPT_DIR / "icon.iconset"
        if src_iconset.exists():
            shutil.copytree(src_iconset, iconset_dir, dirs_exist_ok=True)
        else:
            shutil.copy2(icon_png, resources_dir / "AppIcon.png")
            return "AppIcon"

    # Convert iconset to .icns via iconutil (macOS only)
    icns_path = resources_dir / "AppIcon.icns"
    try:
        run(["iconutil", "-c", "icns", str(iconset_dir), "-o", str(icns_path)])
        # Clean up iconset dir
        shutil.rmtree(iconset_dir, ignore_errors=True)
        print("  Built AppIcon.icns")
        return "AppIcon"
    except (subprocess.CalledProcessError, FileNotFoundError):
        # iconutil not available; fall back to PNG
        shutil.rmtree(iconset_dir, ignore_errors=True)
        shutil.copy2(icon_png, resources_dir / "AppIcon.png")
        print("  Copied AppIcon.png (iconutil unavailable)")
        return "AppIcon"


def create_info_plist(contents_dir, icon_name):
    plist = {
        "CFBundleName": APP_NAME,
        "CFBundleDisplayName": APP_NAME,
        "CFBundleIdentifier": BUNDLE_ID,
        "CFBundleVersion": APP_VERSION,
        "CFBundleShortVersionString": APP_VERSION,
        "CFBundlePackageType": "APPL",
        "CFBundleExecutable": APP_NAME,
        "CFBundleSignature": "????",
        "LSMinimumSystemVersion": "12.3",
        "NSHighResolutionCapable": True,
        "LSApplicationCategoryType": "public.app-category.productivity",
        "LSUIElement": False,
    }
    if icon_name:
        plist["CFBundleIconFile"] = icon_name
    with open(contents_dir / "Info.plist", "wb") as f:
        plistlib.dump(plist, f)


def create_launcher(macos_dir):
    launcher = macos_dir / APP_NAME
    script = f"""#!/bin/bash
# TEI Pipeline Launcher
CONTENTS_DIR="$(cd "$(dirname "$0")/.." && pwd)"
RESOURCES_DIR="$CONTENTS_DIR/Resources"
APP_DIR="$RESOURCES_DIR/app"
PYTHON="$RESOURCES_DIR/venv/bin/python3"
PORT={BACKEND_PORT}

export PYTORCH_ENABLE_MPS_FALLBACK=1
export PYTORCH_MPS_HIGH_WATERMARK_RATIO=0.0

if [ ! -f "$PYTHON" ]; then
    osascript -e 'display alert "TEI Pipeline" message "Python environment not found. Re-run setup_app.py." as critical'
    exit 1
fi

# Kill any previous instance on this port
lsof -ti:$PORT 2>/dev/null | xargs kill -9 2>/dev/null
sleep 0.3

# Start Flask server
cd "$APP_DIR"
"$PYTHON" server.py --port $PORT --no-browser > "$RESOURCES_DIR/server.log" 2>&1 &
SERVER_PID=$!

# Ensure the server dies when the launcher is killed (e.g. Force Quit from Dock)
cleanup() {{
    kill $SERVER_PID 2>/dev/null
    # Also kill any remaining Python processes on our port
    lsof -ti:$PORT 2>/dev/null | xargs kill -9 2>/dev/null
    exit 0
}}
trap cleanup SIGTERM SIGINT SIGHUP EXIT

# Wait for server readiness
READY=0
for i in $(seq 1 60); do
    if curl -s "http://127.0.0.1:$PORT/api/status" > /dev/null 2>&1; then
        READY=1
        break
    fi
    sleep 1
done

if [ "$READY" -eq 1 ]; then
    open "http://127.0.0.1:$PORT"
else
    osascript -e 'display alert "TEI Pipeline" message "Server did not start. Check Resources/server.log for details." as warning'
fi

# Keep running (Dock icon stays). When server exits (shutdown endpoint or crash),
# the wait returns and cleanup runs.
wait $SERVER_PID
"""
    launcher.write_text(script)
    launcher.chmod(0o755)
    print("  Created launcher")


def preflight_check():
    print(f"  Script location: {SCRIPT_DIR}")
    missing = []
    for f in REQUIRED_FILES:
        if not (SCRIPT_DIR / f).is_file():
            missing.append(f)
    for d in REQUIRED_DIRS:
        if not (SCRIPT_DIR / d).is_dir():
            missing.append(d + "/")
    if missing:
        print(f"\n  ERROR: Missing files in {SCRIPT_DIR}:")
        for m in missing:
            print(f"    - {m}")
        sys.exit(1)
    print("  All source files found.")


def copy_source(app_code_dir):
    for f in REQUIRED_FILES:
        shutil.copy2(SCRIPT_DIR / f, app_code_dir / f)
        print(f"    {f}")
    for d in REQUIRED_DIRS:
        shutil.copytree(SCRIPT_DIR / d, app_code_dir / d, dirs_exist_ok=True,
                        ignore=shutil.ignore_patterns("__pycache__", "*.pyc"))
        print(f"    {d}/")


def remove_quarantine(app_dir):
    """Remove the quarantine extended attribute that causes the cancel badge.

    macOS Gatekeeper marks apps created outside of the App Store with
    com.apple.quarantine, which shows a cancel overlay on the icon.
    Removing this xattr + ad-hoc codesigning fixes it.
    """
    try:
        # Remove quarantine from the entire .app tree
        subprocess.run(["xattr", "-dr", "com.apple.quarantine", str(app_dir)],
                       capture_output=True)
        print("  Removed quarantine attribute")
    except FileNotFoundError:
        pass

    try:
        subprocess.run(["codesign", "--force", "--deep", "--sign", "-", str(app_dir)],
                       capture_output=True, check=True)
        print("  Ad-hoc codesigned")
    except (subprocess.CalledProcessError, FileNotFoundError):
        print("  Warning: codesign unavailable. Run manually if cancel badge appears:")
        print(f"    xattr -dr com.apple.quarantine '/Applications/{APP_NAME}.app'")


def main():
    print("=" * 60)
    print(f"  {APP_NAME} Setup")
    print("=" * 60)

    print("\n  Checking source files...")
    preflight_check()

    app_dir = Path("/Applications") / f"{APP_NAME}.app"
    if app_dir.exists():
        print(f"\n  Removing existing {app_dir}...")
        shutil.rmtree(app_dir)

    # 1. Bundle structure + icon
    print("\n[1/6] Creating .app bundle...")
    contents_dir = app_dir / "Contents"
    macos_dir = contents_dir / "MacOS"
    resources_dir = contents_dir / "Resources"
    app_code_dir = resources_dir / "app"
    for d in [macos_dir, app_code_dir]:
        d.mkdir(parents=True, exist_ok=True)

    icon_name = build_icon(resources_dir)
    create_info_plist(contents_dir, icon_name)

    # 2. Venv
    print("\n[2/6] Creating Python virtual environment...")
    venv_dir = resources_dir / "venv"
    run([sys.executable, "-m", "venv", str(venv_dir)])
    pip = str(venv_dir / "bin" / "pip3")
    python = str(venv_dir / "bin" / "python3")
    run([pip, "install", "--upgrade", "pip", "--quiet"])

    # 3. PyTorch
    gpu = detect_platform()
    print(f"\n[3/6] Installing PyTorch (platform: {gpu})...")
    run([pip, "install"] + get_pytorch_install_args(gpu))

    print("\n  Verifying GPU support...")
    run([python, "-c",
         "import torch; "
         "mps = hasattr(torch.backends,'mps') and torch.backends.mps.is_available(); "
         "cuda = torch.cuda.is_available(); "
         "dev = 'mps' if mps else ('cuda' if cuda else 'cpu'); "
         f"print(f'  PyTorch {{torch.__version__}} — device: {{dev}}')"
    ])

    # 4. Other deps
    print(f"\n[4/6] Installing dependencies...")
    deps = [
        "flask>=3.0.0", "flask-cors>=4.0.0",
        "Pillow>=10.0.0", "PyMuPDF>=1.23.0", "lxml>=5.0.0",
        "transformers>=4.40.0", "accelerate>=0.25.0", "qwen-vl-utils>=0.0.4",
    ]
    run([pip, "install"] + deps + ["--quiet"])
    if gpu == "cuda":
        run([pip, "install", "bitsandbytes>=0.41.0", "--quiet"])

    # 5. Copy source
    print(f"\n[5/6] Copying application source...")
    copy_source(app_code_dir)

    # 6. Launcher + signing
    print(f"\n[6/6] Finalizing...")
    create_launcher(macos_dir)
    remove_quarantine(app_dir)

    print("\n" + "=" * 60)
    print(f"  Done! Installed: {app_dir}")
    print(f"  Launch from Applications, Launchpad, or Spotlight.")
    print(f"  Opens in your browser at http://127.0.0.1:{BACKEND_PORT}")
    print(f"  First run downloads model weights (~8 GB).")
    print(f"  To uninstall: delete '{app_dir}'")
    print("=" * 60)


if __name__ == "__main__":
    main()
