#!/usr/bin/env python3
"""
DH Toolkit Launcher — bundled by PyInstaller into a standalone executable.
Starts the Flask server on a free port and opens the default browser.
"""
import sys
import os
import threading
import webbrowser
import socket

# PyInstaller sets sys.frozen and stores bundled files in _MEIPASS
if getattr(sys, 'frozen', False):
    APP_DIR = sys._MEIPASS
    os.chdir(APP_DIR)
else:
    APP_DIR = os.path.dirname(os.path.abspath(__file__))

sys.path.insert(0, APP_DIR)


def find_free_port():
    """Find an available port."""
    with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as s:
        s.bind(('', 0))
        return s.getsockname()[1]


def main():
    port = find_free_port()

    def open_browser():
        import time
        time.sleep(1.5)
        webbrowser.open(f'http://127.0.0.1:{port}')

    threading.Thread(target=open_browser, daemon=True).start()

    from app import app
    print()
    print(f'  DH Toolkit running at http://127.0.0.1:{port}')
    print()
    print('  Your browser should open automatically.')
    print('  Close this window to stop the server.')
    print()
    app.run(host='127.0.0.1', port=port, debug=False, use_reloader=False)


if __name__ == '__main__':
    main()
