#!/usr/bin/env bash
# ═══════════════════════════════════════════════════════════════════════
#  DH Toolkit — macOS Installer (Sonoma 14+ / Sequoia / Tahoe 26)
#  Creates a self-contained .app bundle in ~/Applications.
# ═══════════════════════════════════════════════════════════════════════
set -euo pipefail

APP_NAME="DH Toolkit"
INSTALL_DIR="$HOME/Applications/DH-Toolkit"
VENV_DIR="$INSTALL_DIR/.venv"

RED='\033[0;31m'; GREEN='\033[0;32m'; BLUE='\033[0;34m'
YELLOW='\033[1;33m'; NC='\033[0m'; BOLD='\033[1m'

info()    { echo -e "  ${BLUE}ℹ${NC}  $1"; }
success() { echo -e "  ${GREEN}✓${NC}  $1"; }
warn()    { echo -e "  ${YELLOW}⚠${NC}  $1"; }
fail()    { echo -e "  ${RED}✗${NC}  $1"; exit 1; }

echo ""
echo -e "${BLUE}${BOLD}  ╔═══════════════════════════════════════════════╗${NC}"
echo -e "${BLUE}${BOLD}  ║       DH Toolkit — macOS Installer           ║${NC}"
echo -e "${BLUE}${BOLD}  ╚═══════════════════════════════════════════════╝${NC}"
echo ""

[[ "$(uname)" == "Darwin" ]] || fail "This installer is for macOS only."
success "macOS $(sw_vers -productVersion)"

# ── Find or install Python ───────────────────────────────────────────
PYTHON=""
for cmd in python3.13 python3.12 python3.11 python3.10 python3; do
    if command -v "$cmd" &>/dev/null; then
        v=$("$cmd" -c "import sys;print(f'{sys.version_info.major}.{sys.version_info.minor}')" 2>/dev/null || true)
        if [[ -n "$v" ]]; then
            maj=${v%%.*}; min=${v##*.}
            if (( maj==3 && min>=10 )); then PYTHON="$cmd"; break; fi
        fi
    fi
done

if [[ -z "$PYTHON" ]]; then
    warn "Python 3.10+ not found. Installing via Homebrew..."
    if ! command -v brew &>/dev/null; then
        info "Installing Homebrew..."
        /bin/bash -c "$(curl -fsSL https://raw.githubusercontent.com/Homebrew/install/HEAD/install.sh)"
    fi
    brew install python@3.12
    PYTHON="python3.12"
fi
success "Python: $PYTHON ($($PYTHON --version 2>&1))"

# ── Install app ──────────────────────────────────────────────────────
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
info "Installing to $INSTALL_DIR..."
mkdir -p "$INSTALL_DIR"
cp "$SCRIPT_DIR"/app.py "$SCRIPT_DIR"/passenger_wsgi.py "$SCRIPT_DIR"/requirements.txt "$INSTALL_DIR/"
cp -R "$SCRIPT_DIR"/engines "$INSTALL_DIR/"
cp -R "$SCRIPT_DIR"/templates "$INSTALL_DIR/"
mkdir -p "$INSTALL_DIR/static"
success "Files copied"

info "Creating virtual environment..."
"$PYTHON" -m venv "$VENV_DIR"
"$VENV_DIR/bin/pip" install --upgrade pip -q
"$VENV_DIR/bin/pip" install -r "$INSTALL_DIR/requirements.txt" -q
success "Dependencies installed"

# ── Launcher script ──────────────────────────────────────────────────
cat > "$INSTALL_DIR/launch.command" << 'EOF'
#!/usr/bin/env bash
DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
cd "$DIR"
PORT=${DH_TOOLKIT_PORT:-5000}
if lsof -i ":$PORT" &>/dev/null; then PORT=$((PORT+1)); fi
echo ""
echo "  DH Toolkit starting at http://localhost:$PORT"
echo "  Press Ctrl+C to stop."
echo ""
"$DIR/.venv/bin/python" app.py --host 127.0.0.1 --port "$PORT" 
EOF
chmod +x "$INSTALL_DIR/launch.command"

# ── macOS .app bundle ────────────────────────────────────────────────
APP_BUNDLE="$HOME/Applications/DH Toolkit.app"
mkdir -p "$APP_BUNDLE/Contents/MacOS" "$APP_BUNDLE/Contents/Resources"

cat > "$APP_BUNDLE/Contents/Info.plist" << PLIST
<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE plist PUBLIC "-//Apple//DTD PLIST 1.0//EN"
  "http://www.apple.com/DTDs/PropertyList-1.0.dtd">
<plist version="1.0"><dict>
  <key>CFBundleName</key><string>DH Toolkit</string>
  <key>CFBundleDisplayName</key><string>DH Toolkit</string>
  <key>CFBundleIdentifier</key><string>org.dhtoolkit.app</string>
  <key>CFBundleVersion</key><string>2.0.0</string>
  <key>CFBundleExecutable</key><string>launcher</string>
  <key>CFBundlePackageType</key><string>APPL</string>
  <key>LSMinimumSystemVersion</key><string>14.0</string>
  <key>NSHighResolutionCapable</key><true/>
</dict></plist>
PLIST

cat > "$APP_BUNDLE/Contents/MacOS/launcher" << 'LAUNCHER'
#!/usr/bin/env bash
osascript -e 'tell application "Terminal" to do script "\"$HOME/Applications/DH-Toolkit/launch.command\""'
LAUNCHER
chmod +x "$APP_BUNDLE/Contents/MacOS/launcher"
success "App bundle: $APP_BUNDLE"

echo ""
echo -e "${GREEN}${BOLD}  Installation complete!${NC}"
echo ""
echo "  Double-click 'DH Toolkit' in ~/Applications to launch."
echo "  The app opens in your default web browser."
echo ""
