#!/usr/bin/env bash
# ═══════════════════════════════════════════════════════════════════════
#  DH Toolkit — Linux Installer (from source)
#  Works as both desktop app and cPanel/web server deployment.
# ═══════════════════════════════════════════════════════════════════════
set -euo pipefail

INSTALL_DIR="${DH_TOOLKIT_DIR:-$HOME/dh-toolkit}"
VENV_DIR="$INSTALL_DIR/.venv"

echo ""
echo "  DH Toolkit — Linux Installer"
echo "  ─────────────────────────────"
echo ""

# Find Python 3.10+
PYTHON=""
for cmd in python3.13 python3.12 python3.11 python3.10 python3; do
    if command -v "$cmd" &>/dev/null; then
        v=$("$cmd" -c "import sys;print(f'{sys.version_info.major}.{sys.version_info.minor}')" 2>/dev/null || true)
        if [[ -n "$v" ]]; then
            maj=${v%%.*}; min=${v##*.}
            if (( maj==3 && min>=10 )); then PYTHON="$cmd"; break; fi
        fi
    fi
done
# cPanel CloudLinux paths
if [[ -z "$PYTHON" ]]; then
    for p in /opt/alt/python3{13,12,11,10}/bin/python3; do
        [[ -x "$p" ]] && PYTHON="$p" && break
    done
fi
[[ -n "$PYTHON" ]] || { echo "  ERROR: Python 3.10+ required."; exit 1; }
echo "  Python: $PYTHON ($($PYTHON --version 2>&1))"

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
echo "  Installing to: $INSTALL_DIR"
mkdir -p "$INSTALL_DIR"
cp "$SCRIPT_DIR"/app.py "$SCRIPT_DIR"/passenger_wsgi.py "$SCRIPT_DIR"/requirements.txt "$INSTALL_DIR/"
cp -R "$SCRIPT_DIR"/engines "$INSTALL_DIR/"
cp -R "$SCRIPT_DIR"/templates "$INSTALL_DIR/"
mkdir -p "$INSTALL_DIR/static"

echo "  Creating virtual environment..."
"$PYTHON" -m venv "$VENV_DIR"
"$VENV_DIR/bin/pip" install --upgrade pip -q
"$VENV_DIR/bin/pip" install -r "$INSTALL_DIR/requirements.txt" -q
echo "  Dependencies installed."

# ── Desktop launcher ─────────────────────────────────────────────────
cat > "$INSTALL_DIR/launch.sh" << 'EOF'
#!/usr/bin/env bash
DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
cd "$DIR"
PORT=${DH_TOOLKIT_PORT:-5000}
echo "  DH Toolkit: http://localhost:$PORT"
"$DIR/.venv/bin/python" app.py --host 127.0.0.1 --port "$PORT" --open-browser
EOF
chmod +x "$INSTALL_DIR/launch.sh"

# ── XDG .desktop file (Linux desktop) ────────────────────────────────
if [[ -d "$HOME/.local/share/applications" ]]; then
    cat > "$HOME/.local/share/applications/dh-toolkit.desktop" << DESK
[Desktop Entry]
Name=DH Toolkit
Comment=Digital Humanities Visualization Suite
Exec=$INSTALL_DIR/launch.sh
Terminal=true
Type=Application
Categories=Education;Science;
DESK
    echo "  Desktop entry created."
fi

# ── cPanel / web server support ──────────────────────────────────────
if [[ -d "$HOME/public_html" ]]; then
    echo ""
    echo "  cPanel detected. For web deployment:"
    echo "    1. Setup Python App → Application root: $(basename $INSTALL_DIR)"
    echo "    2. Startup file: passenger_wsgi.py"
    echo "    3. Entry point: application"
    echo "    4. Install deps via the app's terminal: pip install -r requirements.txt"
fi

echo ""
echo "  Installation complete!"
echo "  Run: $INSTALL_DIR/launch.sh"
echo ""
