"""
Geocoding gazetteer — offline coordinate lookup.
Covers ~600+ entries: major world cities, all countries (centroid),
US states, German Länder, Canadian provinces, UK countries/regions,
Australian states, Brazilian states, Indian states, Chinese provinces,
and more political subdivisions.
"""

import re

def _normalize(name: str) -> str:
    """Normalize a place name for matching."""
    s = name.strip().lower()
    s = re.sub(r'[^\w\s]', '', s)
    s = re.sub(r'\s+', ' ', s)
    return s

# ── COUNTRIES (centroid coordinates) ─────────────────────────────────────
COUNTRIES = {
    "afghanistan": (33.93, 67.71), "albania": (41.15, 20.17),
    "algeria": (28.03, 1.66), "andorra": (42.55, 1.60),
    "angola": (-11.20, 17.87), "antigua and barbuda": (17.06, -61.80),
    "argentina": (-38.42, -63.62), "armenia": (40.07, 45.04),
    "australia": (-25.27, 133.78), "austria": (47.52, 14.55),
    "azerbaijan": (40.14, 47.58), "bahamas": (25.03, -77.40),
    "bahrain": (26.07, 50.55), "bangladesh": (23.68, 90.36),
    "barbados": (13.19, -59.54), "belarus": (53.71, 27.95),
    "belgium": (50.50, 4.47), "belize": (17.19, -88.50),
    "benin": (9.31, 2.32), "bhutan": (27.51, 90.43),
    "bolivia": (-16.29, -63.59), "bosnia and herzegovina": (43.92, 17.68),
    "botswana": (-22.33, 24.68), "brazil": (-14.24, -51.93),
    "brunei": (4.54, 114.73), "bulgaria": (42.73, 25.49),
    "burkina faso": (12.24, -1.56), "burundi": (-3.37, 29.92),
    "cabo verde": (16.00, -24.01), "cambodia": (12.57, 104.99),
    "cameroon": (7.37, 12.35), "canada": (56.13, -106.35),
    "central african republic": (6.61, 20.94), "chad": (15.45, 18.73),
    "chile": (-35.68, -71.54), "china": (35.86, 104.20),
    "colombia": (4.57, -74.30), "comoros": (-11.88, 43.87),
    "congo": (-4.04, 21.76), "democratic republic of the congo": (-4.04, 21.76),
    "costa rica": (9.75, -83.75), "croatia": (45.10, 15.20),
    "cuba": (21.52, -77.78), "cyprus": (35.13, 33.43),
    "czech republic": (49.82, 15.47), "czechia": (49.82, 15.47),
    "denmark": (56.26, 9.50), "djibouti": (11.83, 42.59),
    "dominica": (15.41, -61.37), "dominican republic": (18.74, -70.16),
    "ecuador": (-1.83, -78.18), "egypt": (26.82, 30.80),
    "el salvador": (13.79, -88.90), "equatorial guinea": (1.65, 10.27),
    "eritrea": (15.18, 39.78), "estonia": (58.60, 25.01),
    "eswatini": (-26.52, 31.47), "swaziland": (-26.52, 31.47),
    "ethiopia": (9.15, 40.49), "fiji": (-17.71, 178.07),
    "finland": (61.92, 25.75), "france": (46.23, 2.21),
    "gabon": (-0.80, 11.61), "gambia": (13.44, -15.31),
    "georgia": (42.32, 43.36), "germany": (51.17, 10.45),
    "ghana": (7.95, -1.02), "greece": (39.07, 21.82),
    "grenada": (12.26, -61.60), "guatemala": (15.78, -90.23),
    "guinea": (9.95, -9.70), "guinea-bissau": (11.80, -15.18),
    "guyana": (4.86, -58.93), "haiti": (18.97, -72.29),
    "honduras": (15.20, -86.24), "hungary": (47.16, 19.50),
    "iceland": (64.96, -19.02), "india": (20.59, 78.96),
    "indonesia": (-0.79, 113.92), "iran": (32.43, 53.69),
    "iraq": (33.22, 43.68), "ireland": (53.14, -7.69),
    "israel": (31.05, 34.85), "italy": (41.87, 12.57),
    "ivory coast": (7.54, -5.55), "cote d'ivoire": (7.54, -5.55),
    "jamaica": (18.11, -77.30), "japan": (36.20, 138.25),
    "jordan": (30.59, 36.24), "kazakhstan": (48.02, 66.92),
    "kenya": (-0.02, 37.91), "kiribati": (-3.37, -168.73),
    "north korea": (40.34, 127.51), "south korea": (35.91, 127.77),
    "korea": (35.91, 127.77), "kuwait": (29.31, 47.48),
    "kyrgyzstan": (41.20, 74.77), "laos": (19.86, 102.50),
    "latvia": (56.88, 24.60), "lebanon": (33.85, 35.86),
    "lesotho": (-29.61, 28.23), "liberia": (6.43, -9.43),
    "libya": (26.34, 17.23), "liechtenstein": (47.17, 9.56),
    "lithuania": (55.17, 23.88), "luxembourg": (49.82, 6.13),
    "madagascar": (-18.77, 46.87), "malawi": (-13.25, 34.30),
    "malaysia": (4.21, 101.98), "maldives": (3.20, 73.22),
    "mali": (17.57, -4.00), "malta": (35.94, 14.38),
    "marshall islands": (7.13, 171.18), "mauritania": (21.01, -10.94),
    "mauritius": (-20.35, 57.55), "mexico": (23.63, -102.55),
    "micronesia": (7.43, 150.55), "moldova": (47.41, 28.37),
    "monaco": (43.75, 7.41), "mongolia": (46.86, 103.85),
    "montenegro": (42.71, 19.37), "morocco": (31.79, -7.09),
    "mozambique": (-18.67, 35.53), "myanmar": (21.91, 95.96),
    "namibia": (-22.96, 18.49), "nauru": (-0.52, 166.93),
    "nepal": (28.39, 84.12), "netherlands": (52.13, 5.29),
    "new zealand": (-40.90, 174.89), "nicaragua": (12.87, -85.21),
    "niger": (17.61, 8.08), "nigeria": (9.08, 8.68),
    "north macedonia": (41.51, 21.75), "norway": (60.47, 8.47),
    "oman": (21.47, 55.98), "pakistan": (30.38, 69.35),
    "palau": (7.51, 134.58), "palestine": (31.95, 35.23),
    "panama": (8.54, -80.78), "papua new guinea": (-6.31, 143.96),
    "paraguay": (-23.44, -58.44), "peru": (-9.19, -75.02),
    "philippines": (12.88, 121.77), "poland": (51.92, 19.15),
    "portugal": (39.40, -8.22), "qatar": (25.35, 51.18),
    "romania": (45.94, 24.97), "russia": (61.52, 105.32),
    "russian federation": (61.52, 105.32), "rwanda": (-1.94, 29.87),
    "saint kitts and nevis": (17.36, -62.78),
    "saint lucia": (13.91, -60.98), "saint vincent": (12.98, -61.29),
    "samoa": (-13.76, -172.10), "san marino": (43.94, 12.46),
    "saudi arabia": (23.89, 45.08), "senegal": (14.50, -14.45),
    "serbia": (44.02, 21.01), "seychelles": (-4.68, 55.49),
    "sierra leone": (8.46, -11.78), "singapore": (1.35, 103.82),
    "slovakia": (48.67, 19.70), "slovenia": (46.15, 14.99),
    "solomon islands": (-9.65, 160.16), "somalia": (5.15, 46.20),
    "south africa": (-30.56, 22.94), "south sudan": (6.88, 31.31),
    "spain": (40.46, -3.75), "sri lanka": (7.87, 80.77),
    "sudan": (12.86, 30.22), "suriname": (3.92, -56.03),
    "sweden": (60.13, 18.64), "switzerland": (46.82, 8.23),
    "syria": (34.80, 38.99), "taiwan": (23.70, 120.96),
    "tajikistan": (38.86, 71.28), "tanzania": (-6.37, 34.89),
    "thailand": (15.87, 100.99), "timor-leste": (-8.87, 125.73),
    "togo": (8.62, 1.21), "tonga": (-21.18, -175.20),
    "trinidad and tobago": (10.69, -61.22),
    "tunisia": (33.89, 9.54), "turkey": (38.96, 35.24),
    "turkmenistan": (38.97, 59.56), "tuvalu": (-7.11, 177.65),
    "uganda": (1.37, 32.29), "ukraine": (48.38, 31.17),
    "united arab emirates": (23.42, 53.85), "uae": (23.42, 53.85),
    "united kingdom": (55.38, -3.44), "uk": (55.38, -3.44),
    "great britain": (55.38, -3.44),
    "united states": (37.09, -95.71), "united states of america": (37.09, -95.71),
    "usa": (37.09, -95.71), "us": (37.09, -95.71),
    "uruguay": (-32.52, -55.77), "uzbekistan": (41.38, 64.59),
    "vanuatu": (-15.38, 166.96), "vatican city": (41.90, 12.45),
    "venezuela": (6.42, -66.59), "vietnam": (14.06, 108.28),
    "yemen": (15.55, 48.52), "zambia": (-13.13, 27.85),
    "zimbabwe": (-19.02, 29.15),
}

# ── US STATES ────────────────────────────────────────────────────────────
US_STATES = {
    "alabama": (32.32, -86.90), "al": (32.32, -86.90),
    "alaska": (63.59, -154.49), "ak": (63.59, -154.49),
    "arizona": (34.05, -111.09), "az": (34.05, -111.09),
    "arkansas": (35.20, -91.83), "ar": (35.20, -91.83),
    "california": (36.78, -119.42), "ca": (36.78, -119.42),
    "colorado": (39.55, -105.78), "co": (39.55, -105.78),
    "connecticut": (41.60, -73.09), "ct": (41.60, -73.09),
    "delaware": (38.91, -75.53), "de": (38.91, -75.53),
    "florida": (27.66, -81.52), "fl": (27.66, -81.52),
    "georgia": (32.16, -82.90), "ga": (32.16, -82.90),
    "hawaii": (19.90, -155.58), "hi": (19.90, -155.58),
    "idaho": (44.07, -114.74), "id": (44.07, -114.74),
    "illinois": (40.63, -89.40), "il": (40.63, -89.40),
    "indiana": (40.27, -86.13), "in": (40.27, -86.13),
    "iowa": (41.88, -93.10), "ia": (41.88, -93.10),
    "kansas": (39.01, -98.48), "ks": (39.01, -98.48),
    "kentucky": (37.84, -84.27), "ky": (37.84, -84.27),
    "louisiana": (31.17, -91.87), "la": (31.17, -91.87),
    "maine": (45.25, -69.45), "me": (45.25, -69.45),
    "maryland": (39.05, -76.64), "md": (39.05, -76.64),
    "massachusetts": (42.41, -71.38), "ma": (42.41, -71.38),
    "michigan": (44.31, -85.60), "mi": (44.31, -85.60),
    "minnesota": (46.73, -94.69), "mn": (46.73, -94.69),
    "mississippi": (32.35, -89.40), "ms": (32.35, -89.40),
    "missouri": (37.96, -91.83), "mo": (37.96, -91.83),
    "montana": (46.88, -110.36), "mt": (46.88, -110.36),
    "nebraska": (41.49, -99.90), "ne": (41.49, -99.90),
    "nevada": (38.80, -116.42), "nv": (38.80, -116.42),
    "new hampshire": (43.19, -71.57), "nh": (43.19, -71.57),
    "new jersey": (40.06, -74.41), "nj": (40.06, -74.41),
    "new mexico": (34.52, -105.87), "nm": (34.52, -105.87),
    "new york": (40.71, -74.01), "ny": (40.71, -74.01),
    "north carolina": (35.76, -79.02), "nc": (35.76, -79.02),
    "north dakota": (47.55, -101.00), "nd": (47.55, -101.00),
    "ohio": (40.42, -82.91), "oh": (40.42, -82.91),
    "oklahoma": (35.47, -97.52), "ok": (35.47, -97.52),
    "oregon": (43.80, -120.55), "or": (43.80, -120.55),
    "pennsylvania": (41.20, -77.19), "pa": (41.20, -77.19),
    "rhode island": (41.58, -71.48), "ri": (41.58, -71.48),
    "south carolina": (33.84, -81.16), "sc": (33.84, -81.16),
    "south dakota": (43.97, -99.90), "sd": (43.97, -99.90),
    "tennessee": (35.52, -86.58), "tn": (35.52, -86.58),
    "texas": (31.97, -99.90), "tx": (31.97, -99.90),
    "utah": (39.32, -111.09), "ut": (39.32, -111.09),
    "vermont": (44.56, -72.58), "vt": (44.56, -72.58),
    "virginia": (37.43, -78.66), "va": (37.43, -78.66),
    "washington": (47.75, -120.74), "wa": (47.75, -120.74),
    "west virginia": (38.60, -80.45), "wv": (38.60, -80.45),
    "wisconsin": (43.78, -88.79), "wi": (43.78, -88.79),
    "wyoming": (43.08, -107.29), "wy": (43.08, -107.29),
    "district of columbia": (38.91, -77.04), "dc": (38.91, -77.04),
    "washington dc": (38.91, -77.04), "washington d.c.": (38.91, -77.04),
}

# ── GERMAN LÄNDER ────────────────────────────────────────────────────────
GERMAN_STATES = {
    "baden-wurttemberg": (48.66, 9.35), "baden-württemberg": (48.66, 9.35),
    "bavaria": (48.79, 11.50), "bayern": (48.79, 11.50),
    "berlin": (52.52, 13.41),
    "brandenburg": (52.41, 13.06),
    "bremen": (53.08, 8.81),
    "hamburg": (53.55, 9.99),
    "hesse": (50.65, 9.16), "hessen": (50.65, 9.16),
    "lower saxony": (52.64, 9.85), "niedersachsen": (52.64, 9.85),
    "mecklenburg-vorpommern": (53.61, 12.43),
    "north rhine-westphalia": (51.43, 7.66), "nordrhein-westfalen": (51.43, 7.66),
    "rhineland-palatinate": (49.91, 7.45), "rheinland-pfalz": (49.91, 7.45),
    "saarland": (49.40, 7.00),
    "saxony": (51.10, 13.20), "sachsen": (51.10, 13.20),
    "saxony-anhalt": (51.95, 11.69), "sachsen-anhalt": (51.95, 11.69),
    "schleswig-holstein": (54.22, 9.70),
    "thuringia": (50.86, 11.05), "thuringen": (50.86, 11.05), "thüringen": (50.86, 11.05),
}

# ── CANADIAN PROVINCES ───────────────────────────────────────────────────
CANADIAN_PROVINCES = {
    "alberta": (53.93, -116.58), "ab": (53.93, -116.58),
    "british columbia": (53.73, -127.65), "bc": (53.73, -127.65),
    "manitoba": (53.76, -98.81), "mb": (53.76, -98.81),
    "new brunswick": (46.57, -66.46), "nb": (46.57, -66.46),
    "newfoundland and labrador": (53.14, -57.66), "nl": (53.14, -57.66),
    "nova scotia": (44.68, -63.74), "ns": (44.68, -63.74),
    "ontario": (51.25, -85.32), "on": (51.25, -85.32),
    "prince edward island": (46.51, -63.42), "pei": (46.51, -63.42),
    "quebec": (52.94, -73.55), "qc": (52.94, -73.55),
    "saskatchewan": (52.94, -106.45), "sk": (52.94, -106.45),
    "northwest territories": (64.27, -119.18), "nt": (64.27, -119.18),
    "nunavut": (70.30, -83.11), "nu": (70.30, -83.11),
    "yukon": (64.28, -135.00), "yt": (64.28, -135.00),
}

# ── UK REGIONS ───────────────────────────────────────────────────────────
UK_REGIONS = {
    "england": (52.36, -1.17), "scotland": (56.49, -4.20),
    "wales": (52.13, -3.78), "northern ireland": (54.78, -6.49),
}

# ── AUSTRALIAN STATES ────────────────────────────────────────────────────
AUSTRALIAN_STATES = {
    "new south wales": (-31.84, 145.61), "nsw": (-31.84, 145.61),
    "victoria": (-37.47, 144.79), "vic": (-37.47, 144.79),
    "queensland": (-20.92, 142.70), "qld": (-20.92, 142.70),
    "south australia": (-30.00, 136.21), "sa": (-30.00, 136.21),
    "western australia": (-27.67, 121.63), "wa_au": (-27.67, 121.63),
    "tasmania": (-41.45, 145.97), "tas": (-41.45, 145.97),
    "northern territory": (-19.49, 132.55), "nt_au": (-19.49, 132.55),
    "australian capital territory": (-35.47, 149.01), "act": (-35.47, 149.01),
}

# ── INDIAN STATES ────────────────────────────────────────────────────────
INDIAN_STATES = {
    "andhra pradesh": (15.91, 79.74), "arunachal pradesh": (28.22, 94.73),
    "assam": (26.20, 92.94), "bihar": (25.10, 85.31),
    "chhattisgarh": (21.28, 81.87), "goa": (15.30, 74.12),
    "gujarat": (22.26, 71.19), "haryana": (29.06, 76.09),
    "himachal pradesh": (31.10, 77.17), "jharkhand": (23.61, 85.28),
    "karnataka": (15.32, 75.71), "kerala": (10.85, 76.27),
    "madhya pradesh": (22.97, 78.66), "maharashtra": (19.75, 75.71),
    "manipur": (24.66, 93.91), "meghalaya": (25.47, 91.37),
    "mizoram": (23.16, 92.94), "nagaland": (26.16, 94.56),
    "odisha": (20.94, 84.80), "punjab": (31.15, 75.34),
    "rajasthan": (27.02, 74.22), "sikkim": (27.53, 88.51),
    "tamil nadu": (11.13, 78.66), "telangana": (18.11, 79.02),
    "tripura": (23.94, 91.99), "uttar pradesh": (26.85, 80.91),
    "uttarakhand": (30.07, 79.02), "west bengal": (22.99, 87.85),
    "delhi": (28.70, 77.10),
}

# ── CHINESE PROVINCES ────────────────────────────────────────────────────
CHINESE_PROVINCES = {
    "anhui": (31.86, 117.28), "beijing": (39.90, 116.41),
    "chongqing": (29.43, 106.91), "fujian": (26.10, 119.30),
    "gansu": (36.06, 103.83), "guangdong": (23.13, 113.27),
    "guangxi": (22.82, 108.32), "guizhou": (26.65, 106.63),
    "hainan": (20.02, 110.35), "hebei": (38.04, 114.51),
    "heilongjiang": (45.75, 126.65), "henan": (34.77, 113.65),
    "hubei": (30.60, 114.34), "hunan": (28.23, 112.94),
    "inner mongolia": (40.82, 111.65), "jiangsu": (32.06, 118.80),
    "jiangxi": (28.68, 115.89), "jilin": (43.88, 125.32),
    "liaoning": (41.80, 123.43), "ningxia": (38.47, 106.27),
    "qinghai": (36.62, 101.78), "shaanxi": (34.26, 108.94),
    "shandong": (36.67, 116.99), "shanghai": (31.23, 121.47),
    "shanxi": (37.87, 112.55), "sichuan": (30.57, 104.07),
    "tianjin": (39.34, 117.36), "tibet": (29.65, 91.11),
    "xinjiang": (43.79, 87.63), "yunnan": (25.04, 102.71),
    "zhejiang": (30.27, 120.15),
}

# ── BRAZILIAN STATES ─────────────────────────────────────────────────────
BRAZILIAN_STATES = {
    "acre": (-9.97, -67.81), "alagoas": (-9.57, -36.78),
    "amapa": (1.41, -51.77), "amazonas": (-3.07, -61.66),
    "bahia": (-12.58, -41.70), "ceara": (-5.50, -39.32),
    "distrito federal": (-15.83, -47.86), "espirito santo": (-19.18, -40.31),
    "goias": (-15.83, -49.84), "maranhao": (-5.42, -45.44),
    "mato grosso": (-12.64, -55.42), "mato grosso do sul": (-20.77, -54.79),
    "minas gerais": (-18.51, -44.55), "para": (-3.42, -52.49),
    "paraiba": (-7.24, -36.78), "parana": (-24.89, -51.55),
    "pernambuco": (-8.81, -36.95), "piaui": (-7.72, -42.73),
    "rio de janeiro": (-22.91, -43.17), "rio grande do norte": (-5.79, -36.51),
    "rio grande do sul": (-30.03, -51.23), "rondonia": (-11.51, -63.58),
    "roraima": (2.74, -62.08), "santa catarina": (-27.24, -50.22),
    "sao paulo": (-23.55, -46.63), "sergipe": (-10.57, -37.39),
    "tocantins": (-10.25, -48.25),
}

# ── MAJOR CITIES (expanded) ─────────────────────────────────────────────
CITIES = {
    "london": (51.51, -0.13), "paris": (48.86, 2.35),
    "new york": (40.71, -74.01), "new york city": (40.71, -74.01), "nyc": (40.71, -74.01),
    "tokyo": (35.68, 139.65), "beijing": (39.90, 116.41),
    "moscow": (55.76, 37.62), "sydney": (-33.87, 151.21),
    "cairo": (30.04, 31.24), "rio de janeiro": (-22.91, -43.17),
    "mumbai": (19.08, 72.88), "berlin": (52.52, 13.41),
    "rome": (41.90, 12.50), "madrid": (40.42, -3.70),
    "istanbul": (41.01, 28.98), "lagos": (6.52, 3.38),
    "mexico city": (19.43, -99.13), "seoul": (37.57, 126.98),
    "shanghai": (31.23, 121.47), "sao paulo": (-23.55, -46.63),
    "buenos aires": (-34.60, -58.38), "bangkok": (13.76, 100.50),
    "jakarta": (-6.21, 106.85), "nairobi": (-1.29, 36.82),
    "johannesburg": (-26.20, 28.05), "toronto": (43.65, -79.38),
    "chicago": (41.88, -87.63), "los angeles": (34.05, -118.24),
    "la": (34.05, -118.24), "san francisco": (37.77, -122.42),
    "sf": (37.77, -122.42), "singapore": (1.35, 103.82),
    "hong kong": (22.32, 114.17), "taipei": (25.03, 121.57),
    "osaka": (34.69, 135.50), "delhi": (28.70, 77.10),
    "new delhi": (28.61, 77.21), "kolkata": (22.57, 88.36),
    "calcutta": (22.57, 88.36), "karachi": (24.86, 67.00),
    "tehran": (35.69, 51.39), "baghdad": (33.32, 44.37),
    "riyadh": (24.71, 46.68), "ankara": (39.93, 32.86),
    "athens": (37.98, 23.73), "vienna": (48.21, 16.37),
    "warsaw": (52.23, 21.01), "prague": (50.08, 14.44),
    "budapest": (47.50, 19.04), "amsterdam": (52.37, 4.90),
    "brussels": (50.85, 4.35), "lisbon": (38.72, -9.14),
    "dublin": (53.35, -6.26), "edinburgh": (55.95, -3.19),
    "copenhagen": (55.68, 12.57), "stockholm": (59.33, 18.07),
    "oslo": (59.91, 10.75), "helsinki": (60.17, 24.94),
    "reykjavik": (64.15, -21.94), "lima": (-12.05, -77.04),
    "bogota": (4.71, -74.07), "santiago": (-33.45, -70.67),
    "havana": (23.11, -82.37), "casablanca": (33.57, -7.59),
    "algiers": (36.75, 3.06), "tunis": (36.81, 10.18),
    "dakar": (14.72, -17.47), "accra": (5.60, -0.19),
    "addis ababa": (9.03, 38.75), "dar es salaam": (-6.79, 39.21),
    "cape town": (-33.92, 18.42), "manila": (14.60, 120.98),
    "hanoi": (21.03, 105.85), "kuala lumpur": (3.14, 101.69),
    "dhaka": (23.81, 90.41), "colombo": (6.93, 79.86),
    "kabul": (34.56, 69.21), "tashkent": (41.30, 69.24),
    "almaty": (43.22, 76.85), "ulaanbaatar": (47.89, 106.91),
    "yangon": (16.87, 96.20), "phnom penh": (11.56, 104.93),
    "wellington": (-41.29, 174.78), "auckland": (-36.85, 174.76),
    "honolulu": (21.31, -157.86), "anchorage": (61.22, -149.90),
    "vancouver": (49.28, -123.12), "montreal": (45.50, -73.57),
    "washington dc": (38.91, -77.04), "washington": (38.91, -77.04),
    "boston": (42.36, -71.06), "miami": (25.76, -80.19),
    "houston": (29.76, -95.37), "denver": (39.74, -104.99),
    "seattle": (47.61, -122.33), "phoenix": (33.45, -112.07),
    "atlanta": (33.75, -84.39), "detroit": (42.33, -83.05),
    "minneapolis": (44.98, -93.27), "st louis": (38.63, -90.20),
    "saint louis": (38.63, -90.20), "new orleans": (29.95, -90.07),
    "philadelphia": (39.95, -75.17), "pittsburgh": (40.44, -79.99),
    "portland": (45.52, -122.68), "san diego": (32.72, -117.16),
    "dallas": (32.78, -96.80), "zurich": (47.38, 8.54),
    "geneva": (46.20, 6.14), "munich": (48.14, 11.58),
    "munchen": (48.14, 11.58), "frankfurt": (50.11, 8.68),
    "milan": (45.46, 9.19), "milano": (45.46, 9.19),
    "barcelona": (41.39, 2.17), "seville": (37.39, -5.98),
    "lyon": (45.76, 4.84), "marseille": (43.30, 5.37),
    "hamburg": (53.55, 9.99), "florence": (43.77, 11.26),
    "firenze": (43.77, 11.26), "venice": (45.44, 12.32),
    "venezia": (45.44, 12.32), "naples": (40.85, 14.27),
    "napoli": (40.85, 14.27), "st petersburg": (59.93, 30.34),
    "saint petersburg": (59.93, 30.34), "kyiv": (50.45, 30.52),
    "kiev": (50.45, 30.52), "bucharest": (44.43, 26.10),
    "belgrade": (44.79, 20.45), "sofia": (42.70, 23.32),
    "zagreb": (45.82, 15.98), "sarajevo": (43.86, 18.41),
    "beirut": (33.89, 35.50), "amman": (31.95, 35.93),
    "jerusalem": (31.77, 35.21), "tel aviv": (32.09, 34.78),
    "dubai": (25.20, 55.27), "doha": (25.29, 51.53),
    "muscat": (23.59, 58.38), "kuwait city": (29.38, 47.98),
    "baku": (40.41, 49.87), "tbilisi": (41.72, 44.83),
    "yerevan": (40.18, 44.50), "guangzhou": (23.13, 113.26),
    "shenzhen": (22.54, 114.06), "chengdu": (30.57, 104.07),
    "wuhan": (30.59, 114.31), "nanjing": (32.06, 118.80),
    "hangzhou": (30.27, 120.16), "kyoto": (35.01, 135.77),
    "hiroshima": (34.39, 132.46), "busan": (35.18, 129.08),
    "lahore": (31.52, 74.36), "islamabad": (33.68, 73.05),
    "chennai": (13.08, 80.27), "bangalore": (12.97, 77.59),
    "bengaluru": (12.97, 77.59), "hyderabad": (17.39, 78.49),
    "ahmedabad": (23.02, 72.57), "pune": (18.52, 73.86),
    "jaipur": (26.91, 75.79), "austin": (30.27, -97.74),
    "nashville": (36.16, -86.78), "san antonio": (29.42, -98.49),
    "columbus": (39.96, -82.99), "charlotte": (35.23, -80.84),
    "indianapolis": (39.77, -86.16), "jacksonville": (30.33, -81.66),
    "san jose": (37.34, -121.89), "fort worth": (32.75, -97.33),
    "baltimore": (39.29, -76.61), "milwaukee": (43.04, -87.91),
    "las vegas": (36.17, -115.14), "salt lake city": (40.76, -111.89),
    "memphis": (35.15, -90.05), "oklahoma city": (35.47, -97.52),
    "louisville": (38.25, -85.76), "richmond": (37.54, -77.44),
    "hartford": (41.76, -72.68), "raleigh": (35.78, -78.64),
    "birmingham": (52.49, -1.90), "manchester": (53.48, -2.24),
    "glasgow": (55.86, -4.25), "liverpool": (53.41, -2.98),
    "leeds": (53.80, -1.55), "bristol": (51.45, -2.59),
    "cardiff": (51.48, -3.18), "belfast": (54.60, -5.93),
    "cologne": (50.94, 6.96), "koln": (50.94, 6.96),
    "dusseldorf": (51.23, 6.78), "stuttgart": (48.78, 9.18),
    "dortmund": (51.51, 7.47), "essen": (51.46, 7.01),
    "leipzig": (51.34, 12.37), "dresden": (51.05, 13.74),
    "nuremberg": (49.45, 11.08), "nurnberg": (49.45, 11.08),
    "hannover": (52.37, 9.74), "hanover": (52.37, 9.74),
    "toulouse": (43.60, 1.44), "nice": (43.71, 7.26),
    "nantes": (47.22, -1.55), "strasbourg": (48.57, 7.75),
    "bordeaux": (44.84, -0.58), "lille": (50.63, 3.06),
    "montpellier": (43.61, 3.88), "osaka": (34.69, 135.50),
    "nagoya": (35.18, 136.91), "sapporo": (43.06, 141.35),
    "fukuoka": (33.59, 130.40), "kobe": (34.69, 135.20),
    "yokohama": (35.44, 139.64),
    "perth": (-31.95, 115.86), "brisbane": (-27.47, 153.03),
    "melbourne": (-37.81, 144.96), "adelaide": (-34.93, 138.60),
    "canberra": (-35.28, 149.13), "hobart": (-42.88, 147.33),
    "ottawa": (45.42, -75.70), "calgary": (51.05, -114.07),
    "edmonton": (53.55, -113.49), "winnipeg": (49.90, -97.14),
    "quebec city": (46.81, -71.21),
}


def geocode(text: str):
    """
    Attempt to geocode a place string.
    Tries exact match, then partial match, then parses composite addresses.
    Returns (lat, lon) or None.
    """
    if not text or not text.strip():
        return None

    key = _normalize(text)

    # Direct lookup in order of specificity
    for gazetteer in [CITIES, US_STATES, GERMAN_STATES, CANADIAN_PROVINCES,
                      UK_REGIONS, AUSTRALIAN_STATES, INDIAN_STATES,
                      CHINESE_PROVINCES, BRAZILIAN_STATES, COUNTRIES]:
        if key in gazetteer:
            return gazetteer[key]

    # Try parsing composite address: "City, State, Country" or "City, Country"
    parts = [p.strip() for p in text.split(',') if p.strip()]
    for part in parts:
        pk = _normalize(part)
        for gazetteer in [CITIES, US_STATES, GERMAN_STATES, CANADIAN_PROVINCES,
                          UK_REGIONS, AUSTRALIAN_STATES, INDIAN_STATES,
                          CHINESE_PROVINCES, BRAZILIAN_STATES, COUNTRIES]:
            if pk in gazetteer:
                return gazetteer[pk]

    # Partial/substring match (last resort)
    for gazetteer in [CITIES, COUNTRIES]:
        for k, v in gazetteer.items():
            if key in k or k in key:
                return v

    return None
