#!/usr/bin/env bash
# ═══════════════════════════════════════════════════════════════════════
#  DH Toolkit — Build macOS .app bundle and .dmg installer
#  Run this on a macOS machine with Python 3.10+.
#  Output: dist/DH-Toolkit.dmg (drag-to-Applications installer)
# ═══════════════════════════════════════════════════════════════════════
set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
BUILD_DIR="$SCRIPT_DIR/build_mac"
DIST_DIR="$SCRIPT_DIR/dist"

echo ""
echo "  Building DH Toolkit macOS application..."
echo ""

# ── Ensure build tools ───────────────────────────────────────────────
pip3 install pyinstaller --break-system-packages 2>/dev/null || pip3 install pyinstaller

# ── Create the launcher that starts Flask and opens browser ──────────
# (launcher.py is already in the project)

# ── Run PyInstaller ──────────────────────────────────────────────────
cd "$SCRIPT_DIR"

pyinstaller \
    --name "DH Toolkit" \
    --windowed \
    --onedir \
    --noconfirm \
    --clean \
    --add-data "templates:templates" \
    --add-data "static:static" \
    --add-data "engines:engines" \
    --add-data "app.py:." \
    --hidden-import=engines.gazetteer \
    --hidden-import=engines.network \
    --hidden-import=engines.lda \
    --hidden-import=flask \
    --hidden-import=numpy \
    --hidden-import=PIL \
    --hidden-import=PIL.Image \
    --hidden-import=PIL.ImageDraw \
    --hidden-import=PIL.ImageFont \
    --collect-submodules=numpy \
    --collect-submodules=PIL \
    launcher.py

echo "  .app bundle built at: $DIST_DIR/DH Toolkit.app"

# ── Create DMG ───────────────────────────────────────────────────────
DMG_NAME="DH-Toolkit-Installer"
DMG_PATH="$DIST_DIR/$DMG_NAME.dmg"
DMG_STAGING="$BUILD_DIR/dmg_staging"

rm -rf "$DMG_STAGING"
mkdir -p "$DMG_STAGING"

# Copy .app to staging
cp -R "$DIST_DIR/DH Toolkit.app" "$DMG_STAGING/"

# Create Applications symlink for drag-to-install
ln -s /Applications "$DMG_STAGING/Applications"

# Create DMG
if command -v hdiutil &>/dev/null; then
    rm -f "$DMG_PATH"
    hdiutil create \
        -volname "DH Toolkit" \
        -srcfolder "$DMG_STAGING" \
        -ov \
        -format UDZO \
        "$DMG_PATH"
    echo ""
    echo "  ✓ DMG created: $DMG_PATH"
    echo "  Users can double-click the DMG and drag 'DH Toolkit' to Applications."
else
    echo "  hdiutil not available (not on macOS?). .app is at: $DIST_DIR/DH Toolkit.app"
fi

# Cleanup temp files
rm -rf "$DMG_STAGING"

echo ""
echo "  Build complete!"
echo ""
